/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

static const char __idstring[] = "@(#)$Id: mx_cancel.c,v 1.48 2006/09/13 02:23:15 bgoglin Exp $";

#include "mx_auto_config.h"
#include "myriexpress.h"
#include "mx__lib_types.h"
#include "mx__request.h"
#include "mx__error.h"
#include "mx__endpoint.h"
#include "mx__debug_dump.h"

int mx__test_common(mx_endpoint_t ep, mx_request_t *request,
		    mx_status_t *status);

MX_FUNC(mx_return_t)
mx_cancel(mx_endpoint_t endpoint, mx_request_t *reqp, uint32_t *result)
{
  union mx_request *req = *reqp;
  mx_return_t ret = MX_SUCCESS;

  MX__MUTEX_LOCK(&endpoint->lock);
  /* Search in the send request queue and recv request queue. */

  switch (req->basic.type) {
  case MX__REQUEST_TYPE_RECV: {
    if (req->recv.basic.state & MX__REQUEST_STATE_RECV_MATCHED) {
      /* already matched, too late */
      *result = 0;
    } else {
      /* not matched, still in the recv queue */
      uint32_t ctxid = CTXID_FROM_MATCHING(endpoint, req->recv.match_info);
      mx__spliceout_request(&endpoint->ctxid[ctxid].recv_reqq, req);
      if (req->recv.count > 1) {
	/* release the segment list */
	mx_free(req->recv.segments);
      }
      mx__rl_free(endpoint, req);
      *reqp = 0;
      *result = 1;
    }
    break;
  }

  case MX__REQUEST_TYPE_RECV_LARGE:
    /* RECV are converted to RECV_LARGE when matched, so it's already too late */
    *result = 0;
    break;

  case MX__REQUEST_TYPE_CONNECT:
    while (req->connect.basic.state & MX__REQUEST_STATE_MCP) {
      /* busy loop until the request is out of MCP send queue */
      mx__luigi(endpoint);
    }

    if (req->connect.basic.state & MX__REQUEST_STATE_COMPLETED) {
      /* the request is already completed */
      *result = 0;
    } else {
      /* the request is pending on a queue */
      struct mx__request_queue_head * head = &endpoint->resend_list;
      if (req->connect.basic.state & MX__REQUEST_STATE_SEND_QUEUED)
	head = req->connect.basic.requeued ? &endpoint->resend_reqq : &endpoint->send_reqq;

      mx__spliceout_request(head, req);
      mx__rl_free(endpoint, req);
      *reqp = 0;
      *result = 1;
    }
    break;

  default:
    /* SEND_* are NOT cancellable with mx_cancel() */
    ret = mx__error(endpoint, "mx_cancel", MX_CANCEL_NOT_SUPPORTED);
  }

  MX__MUTEX_UNLOCK(&endpoint->lock);
  return ret;
}

MX_FUNC(mx_return_t)
mx_forget(mx_endpoint_t endpoint, mx_request_t *reqp)
{
  mx_status_t status;

  MX__MUTEX_LOCK(&endpoint->lock);

  /* either the request has completed and we test it,
   * or we mark is as forgettable
   */
  if (!mx__test_common(endpoint, reqp, &status))
    (*reqp)->basic.state |= MX__REQUEST_STATE_DEAD;
    /* FIXME: ep->dead_count++ ? */

  MX__MUTEX_UNLOCK(&endpoint->lock);
  return MX_SUCCESS;
}
